function [EMISSIONS, EMISSION_SOURCE] = GHGM2_greenhouse_gas_calculation(dairyMS, SheepSR, ForestAge, ScrubAge, LURNZDATA, c, coef, last_year, lookup_tables, ALL_YEARS )
%
% Greenhouse Gas Module - function 2 (GHGM2)
%
% This file calculates Greenhouse Gas emissions as CO2-equivalents
% according to coefficients provided by Timar.
%
%   13-Jan-12 : Dairy and Sheep/beef emissions only
%

% Code by Simon Anastasiadis : 2012-04-05

%% Parameters

% base year
base_year = 2008;

%% Identify years of interest

% construct vector containing years of interest
if ALL_YEARS
    % we want all years
    years = (base_year:last_year).';
else
    % we want just the final year
    years = last_year;
end

%% Storage for Emissions

% Dairy
EMISSIONS.dairy  = zeros(size(dairyMS));
EMISSION_SOURCE.dairy_milk = zeros(size(dairyMS));
EMISSION_SOURCE.dairy_meat = zeros(size(dairyMS));
EMISSION_SOURCE.dairy_fert = zeros(size(dairyMS));
% Sheep/Beef
EMISSIONS.sheep  = zeros(size(SheepSR));
EMISSION_SOURCE.sheep_meat = zeros(size(SheepSR));
EMISSION_SOURCE.beef_meat  = zeros(size(SheepSR));
EMISSION_SOURCE.sbeef_fert = zeros(size(SheepSR));
% Forestry
EMISSIONS.forest = zeros(size(ForestAge));
% Scrub
EMISSIONS.scrub  = zeros(size(ScrubAge));

%% Calculate dairy emissions

% coefficient holders
dairy_GHG_per_milksolid   = zeros(1,size(dairyMS,2));
dairy_GHG_per_stock       = zeros(1,size(dairyMS,2));
dairy_stock_per_ha        = zeros(size(dairyMS));
dairy_GHG_per_Nfertiliser = zeros(1,size(dairyMS,2));
dairy_Nfertiliser_per_milksolid  = zeros(1,size(dairyMS,2));
dairy_scaling_coef        = zeros(1,size(dairyMS,2));

% unique LIC regions
LIC_regions = unique(coef.dairy_stock_per_ha(:,1));

% fill coefficients
for ii = 2:size(dairyMS,2)
    % dairy coefficients
    dairy_GHG_per_milksolid(ii)   = formula_or_constant(coef.dairy_GHG_per_milksolid, years(ii-1));
    dairy_GHG_per_stock(ii)       = formula_or_constant(coef.dairy_GHG_per_stock, years(ii-1));
    dairy_GHG_per_Nfertiliser(ii) = formula_or_constant(coef.dairy_GHG_per_Nfertiliser, years(ii-1));
    dairy_Nfertiliser_per_milksolid(ii)  = formula_or_constant(coef.dairy_Nfertiliser_per_milksolid, years(ii-1));
    dairy_scaling_coef(ii)        = formula_or_constant(coef.dairy_scaling_coef, years(ii-1));
    
    % dairy stock_per_ha by LIC regions
    for jj = LIC_regions'
        % indicator in LURNZDATA
        iLIC = LURNZDATA(:,c.lic) == jj;
        % indicator in dairy_stock_per_ha
        iCOEFS = coef.dairy_stock_per_ha(:,1) == jj;
        % do the assignment
        dairy_stock_per_ha(iLIC,ii) = formula_or_constant(coef.dairy_stock_per_ha(iCOEFS,2:end), years(ii-1));
    end
end

% indicator for dairy
iDairy = dairyMS > 0;
% emissions only occur on those pixels with dairy production

% calculate dairy emissions as sum of milk, meat and fertiliser emissions
EMISSION_SOURCE.dairy_milk = (ones(size(dairyMS,1),1) * dairy_GHG_per_milksolid) .* dairyMS;
EMISSION_SOURCE.dairy_meat = (ones(size(dairyMS,1),1) * (dairy_GHG_per_stock .* dairy_scaling_coef)) .* dairy_stock_per_ha .* iDairy;
EMISSION_SOURCE.dairy_fert = (ones(size(dairyMS,1),1) * (dairy_GHG_per_Nfertiliser .* dairy_Nfertiliser_per_milksolid)) .* dairyMS;

EMISSIONS.dairy = EMISSION_SOURCE.dairy_milk + EMISSION_SOURCE.dairy_meat + EMISSION_SOURCE.dairy_fert;
% note: scaling not used for milk and fertiliser emissions because it is
% already used in the calculation of dairyMS
%
%% Calculate sheep/beef emissions

% coefficient holders
SB_sheep_SU_proportion = zeros(size(SheepSR));
SB_GHG_per_sheep_SU    = zeros(1,size(SheepSR,2));
SB_GHG_per_beef_SU     = zeros(1,size(SheepSR,2));
SB_GHG_per_Nfertiliser = zeros(1,size(SheepSR,2));
SB_Nfertiliser_per_SU  = zeros(1,size(SheepSR,2));
SB_scaling_coef        = zeros(1,size(SheepSR,2));

% unique SB regions
SB_regions = unique(coef.SB_sheep_SU_proportion(:,1));

% fill coefficients
for ii = 2:size(dairyMS,2)
    % sheep/beef coefficients
    SB_GHG_per_sheep_SU(ii)    = formula_or_constant(coef.SB_GHG_per_sheep_SU, years(ii-1));
    SB_GHG_per_beef_SU(ii)     = formula_or_constant(coef.SB_GHG_per_beef_SU, years(ii-1));
    SB_GHG_per_Nfertiliser(ii) = formula_or_constant(coef.SB_GHG_per_Nfertiliser, years(ii-1));
    SB_Nfertiliser_per_SU(ii)  = formula_or_constant(coef.SB_Nfertiliser_per_SU, years(ii-1));
    SB_scaling_coef(ii)        = formula_or_constant(coef.SB_scaling_coef, years(ii-1));
    
    % Sheep SU proportion by regions
    for jj = SB_regions'
        % indicator in LURNZDATA
        iCLASS = LURNZDATA(:,c.sbclass) == jj;
        % indicator in SB_sheep_SU_proportion
        iCOEFS = coef.SB_sheep_SU_proportion(:,1) == jj;
        % do the assignment
        SB_sheep_SU_proportion(iCLASS,ii) = formula_or_constant(coef.SB_sheep_SU_proportion(iCOEFS,2:end), years(ii-1));
    end
end

% indicator for sheep
% iSheep = SheepSR > 0;
% emissions only occur on those pixels with sheep / beef stock units

EMISSION_SOURCE.sheep_meat = (ones(size(SheepSR,1),1) * SB_GHG_per_sheep_SU) .* SB_sheep_SU_proportion .* SheepSR;
EMISSION_SOURCE.beef_meat  = (ones(size(SheepSR,1),1) * SB_GHG_per_beef_SU) .* (1-SB_sheep_SU_proportion) .* SheepSR;
EMISSION_SOURCE.sbeef_fert = (ones(size(SheepSR ,1),1) * (SB_GHG_per_Nfertiliser .* SB_Nfertiliser_per_SU ./ SB_scaling_coef)) .* SheepSR;
% note: scaling coefficient needed for fertiliser emissions (to convert 
% back from stocking rate to CCAV)

EMISSIONS.sheep = EMISSION_SOURCE.sheep_meat + EMISSION_SOURCE.beef_meat + EMISSION_SOURCE.sbeef_fert;
%
%% Calculate Forestry emissions (and sequestration)
% We use negative ages to indicate harvest and deforestation

% setup LURNZDATA
LURNZDATA = setup_forestry_Data(LURNZDATA, c);

% set as matrices
forest_regions = LURNZDATA(:,c.ta)    * ones(1,size(ForestAge,2));
forest_LUCAS   = LURNZDATA(:,c.LUCAS) * ones(1,size(ForestAge,2));
% indicator for pixels of interest
iForest = ForestAge ~= -9999 & forest_regions ~= 999;

% forestry emissions
EMISSIONS.forest(iForest) = lookup_forest_emissions(ForestAge(iForest), forest_regions(iForest), forest_LUCAS(iForest), lookup_tables);

% remove unneeded data
clear('LURNZDATA', 'forest_LUCAS', 'forest_regions', 'iForest')

%% Calculate Scrub sequestration
% Unlike forestry there are no negative ages

% indicator for pixels of interest
iScrub = ScrubAge ~= -9999;

% Scrub emissions
EMISSIONS.scrub(iScrub) = lookup_scrub_emissions(ScrubAge(iScrub), lookup_tables);

% remove unneeded data
clear('iForest')


%% Complete EMISSIONS

% Calculate total emissions
EMISSIONS.total = EMISSIONS.dairy + EMISSIONS.sheep + EMISSIONS.forest + EMISSIONS.scrub;
% insert sorters
EMISSIONS.dairy(:,1)  = dairyMS(:,1);
EMISSIONS.sheep(:,1)  = SheepSR(:,1);
EMISSIONS.forest(:,1) = ForestAge(:,1);
EMISSIONS.scrub(:,1)  = ScrubAge(:,1);
EMISSIONS.total(:,1)  = SheepSR(:,1);
% all sorters should be the same, but this gives us a way to check if
% something goes wrong

%% Inform User projecting land use change is complete

msg = sprintf(' - - GHGM2 : Greenhouse Gas Emissions Calculation complete');
disp(msg)

end

%% Subfunction : Formula Value or Constant

function [value ] = formula_or_constant(coef, year)
% either length(coef) = 3
% 1. alpha
% 2. beta
% 3. gamma
% OR length(coef) = 1
% 1. constant

% Value = alpha + beta * Ln(year - gamma) or constant (whichever is provided).

% use constant if it is provided
if length(coef) == 1
    value = coef;
% use constant is first term is NaN
elseif isnan(coef(1)) && ~isnan(coef(4))
    value = coef(4);
% otherwise use formula
else
    alpha = coef(1);
    beta = coef(2);
    gamma = coef(3);
    value = alpha + beta * log(year - gamma);
end

end

%% Subfunction : Calculation Forestry Carbon Emissions & Sequestration

function [LURNZDATA ] = setup_forestry_Data(LURNZDATA, c)
%
% Convert the TAs into forestry regions

% Convert TAs into regions
% overwriting TAs using negative numbers to avoid confusion
% 1 Auckland
iTemp = 1 <= LURNZDATA(:,c.ta) & LURNZDATA(:,c.ta) <= 15;
LURNZDATA(iTemp,c.ta) = -1;
% 2 Waikato / Taupo
iTemp = ((16 <= LURNZDATA(:,c.ta) & LURNZDATA(:,c.ta) <= 21) | LURNZDATA(:,c.ta) == 36);
LURNZDATA(iTemp,c.ta) = -2;
% 3 Bay of Plenty
iTemp = 22 <= LURNZDATA(:,c.ta) & LURNZDATA(:,c.ta) <= 27;
LURNZDATA(iTemp,c.ta) = -3;
% 4 Gisborne
iTemp = LURNZDATA(:,c.ta) == 28;
LURNZDATA(iTemp,c.ta) = -4;
% 5 Hawkes Bay / South North Island
iTemp = (29 <= LURNZDATA(:,c.ta) & LURNZDATA(:,c.ta) <= 35) | (37 <= LURNZDATA(:,c.ta) & LURNZDATA(:,c.ta) <= 50);
LURNZDATA(iTemp,c.ta) = -5;
% 6 Nelson / Marlborough
iTemp = 51 <= LURNZDATA(:,c.ta) & LURNZDATA(:,c.ta) <= 54;
LURNZDATA(iTemp,c.ta) = -6;
% 7 Canterbury / West Coast
iTemp = 55 <= LURNZDATA(:,c.ta) & LURNZDATA(:,c.ta) <= 66;
LURNZDATA(iTemp,c.ta) = -7;
% 8 Otago
iTemp = 68 <= LURNZDATA(:,c.ta) & LURNZDATA(:,c.ta) <= 71;
LURNZDATA(iTemp,c.ta) = -8;
% 9 Southland
iTemp = 72 <= LURNZDATA(:,c.ta) & LURNZDATA(:,c.ta) <= 75;
LURNZDATA(iTemp,c.ta) = -9;
% swap regions from neative to possitive
LURNZDATA(:,c.ta) = -LURNZDATA(:,c.ta);

end

%% Subfunction : Lookup Emissions from Table

function [forestry_emissions ] = lookup_forest_emissions(ForestAge, region, LUCAS, lookup_tables)
% Given the change in the age of all the forestry we calculate the implied
% emissions (from harvesting) and sequestration (from growth) for all years
% of interest according to the MAF lookup tables.

% LUCAS code 73 is pre90

% Create empty vector to fill
forestry_emissions = zeros(size(ForestAge));

% minimum forestry age
% pre90 forest = 10 years
% post89 forest = 0 years
% maximum forestry age = 50 years
ForestAge = sign(ForestAge) .* min(abs(ForestAge),50);
% we use sign() and abs() to handle harvesting and deforestation which are
% given as negative numbers

% loop through all forestry pixels
for ii = 1:length(ForestAge)
    % select correct table
    if LUCAS(ii) == 73 && ForestAge(ii) > 10
        % pre 1990
        lookup_tab = lookup_tables.pre90;
    else
        % post 1989
        lookup_tab = lookup_tables.post89;
    end
    
    % identify row and colunm of interest
    row = abs(ForestAge(ii)) == lookup_tab(:,1);
    col = region(ii) + 1;
    
    % forest at maximum age
    if ForestAge(ii) == 50
        forestry_emissions(ii) = 0;
    % harvest and deforestation
    elseif ForestAge(ii) < 0 && col > 0
        % look up & scale as data file is in tonnes
        forestry_emissions(ii) = lookup_tab(row,col) * 1000;
    % forestry aging
    elseif ForestAge(ii) > 0 && col > 0
        % look up & scale as data file is in tonnes
        try
            forestry_emissions(ii) = (lookup_tab(find(row)-1,col) - lookup_tab(row,col)) * 1000;
        catch
            disp('there has been an error, please fix it')
        end
    end
end

end

%% Subfunction : Lookup Emissions from Table

function [scrub_sequestration ] = lookup_scrub_emissions(ScrubAge, lookup_tables)
% Given the change in the age of all scrub we calculate the implied
% sequestration (from growth) for all years of interest according to the
% MAF lookup tables.

% Create empty vector to fill
scrub_sequestration = zeros(size(ScrubAge));

% maximum scrub age = 50 years
ScrubAge = min(abs(ScrubAge),50);

% loop through all forestry pixels
for ii = 1:length(ScrubAge)
    % select correct table
    % only use the post89 table (as new Scrub)
    lookup_tab = lookup_tables.post89;
    
    % identify row and colunm of interest
    row = abs(ScrubAge(ii)) == lookup_tab(:,1);
    col = 11;
    
    % scrub at maximum age
    if ScrubAge(ii) == 50
        scrub_sequestration(ii) = 0;
    % scrub aging
    elseif ScrubAge(ii) > 0
        % look up & scale as data file is in tonnes
        try
            scrub_sequestration(ii) = (lookup_tab(find(row)-1,col) - lookup_tab(row,col)) * 1000;
        catch
            disp('hi')
        end
    end
end

end
